//=============================================================================
//
//  ShiranganaConverter.js   しらんがな音声変換
//                             Programmed by G.K.2022.6.21
//
//=============================================================================

//変数の宣言===================================================================
var canvas;				    //キャンバスオブジェクト
var ctx;				    //描画コンテキスト

var winWidth = 1200;		//CANVASの横幅
var winHeight = 800;		//CANVASの高さ

var mode = 0;

var src_bytes = [];         // バイト配列を格納する変数
var src_int_Left = [];      // 整数変換されたデータチャンク 左側
var src_int_Right = [];     //                              右側

var dst_bytes = [];         // 変換後のWAVデータ配列

var dft_Re = [];            // [出力] 実数部
var dft_Im = [];            // [出力] 虚数部
var graphZoom = 1;          // グラフの倍率

var dft_Re_Wave = [];       // [出力] 実数部
var dft_Im_Wave = [];       // [出力] 虚数部

var wave_diff = [];         // 微分データ
var graphZoom2 = 0;         

var cropStart;              // 切り取り開始位置
var cropEnd;                // 切り取り終了位置
var cropLength;             // 切り取り長さ
var backLength;             // 切り取り～最後尾までのサイズ
var wordSeparate = [];      // 語節切り分け位置


//ヘッダーデータ
var Head_RIFF;              //固定文字RIFF
var src_length = 0;         //元ファイルのサイズ
var Head_WAVEfmt;           //固定文字WAVEfmt_

var Head_LinearPCM;         //リニアPCM値
var Head_Format;            //フォーマット
var Head_Channel;           //チャネル数
var Head_Sampling;          //サンプリング周波数
var Head_ByteSec;           //秒間バイト数

var Head_BlockSize;         //ブロックサイズ
var Head_BitSample;         //ビット／サンプル値
var Head_DIFdata;           //固定文字data
var Head_ChunkSize;         //チャンクサイズ

var FLG_LoadSound;          //音声データ読み込み完了フラグ
var FLG_HeaderCheck;        //ヘッダーチェック実行フラグ
var FLG_CalcDFT;            //フーリエ変換済み
var FLG_CropData;           //データ切り取り済み
var FLG_ConvExec;           //変換完了


//関数の宣言========================================================================


//初期化処理-------------------------------------------
function init() {

    //描画コンテキストの取得
    canvas = document.getElementById("mainCanvas");
    if ( ! canvas || ! canvas.getContext ) { return false; }
    ctx = canvas.getContext('2d');

    FLG_LoadSound = false;
    FLG_HeaderCheck = false;
    FLG_CalcDFT = false;
    FLG_CropData = false;

    cropStart = -1;
    cropEnd = -1;
    cropLength = -1;

    ChangeControlMode(0);

    draw();
}


//画面描画--------------------------------------------
function draw()
{
    ctx.font = "16px 'ＭＳ Ｐゴシック'";
    ctx.strokeStyle = "rgba(0,0,0,1)";
    ctx.lineWidth = 1;
    ctx.clearRect(0, 0, winWidth, winHeight);		//画面のクリア

    switch(mode)
    {
        //WAVファイル未読み込み
        case 0:
            ctx.fillText("[1] WAVファイルを選択してください。", 20, 20);
            break;

        //切り取り未完了
        case 1:
            ctx.fillText("[2] 前後の余計な部分を切り取ります。", 20, 20);

            CheckHeaderData();
            ConvertDataByteToInt();
          
            if(FLG_CalcDFT == false)
            {
                //CulcMean();                
                CalcDiff();


                //dft();
                Calc_SDFT();
            }

            Disp_Header();            
            Disp_Spectgram();
            Disp_DiffWave();
            Disp_CropLine();
            break;

        //切り取り完了後
        case 2:
            ctx.fillText("[3] 各単語の区切りを指定します。", 20, 20);

            if(FLG_CalcDFT == false)
            {
                Calc_SDFT();
            }

            Disp_Header();
            Disp_Spectgram();
            Disp_DiffWave();

            Disp_SeparatePosition();

            break;

        //音声変換完了後
        case 3:
            ctx.fillText("[4] WAVデータをダウンロードします。", 20, 20);
            break;

    }

}


//モードに合わせてコントロールの無効を切替 ------------
function ChangeControlMode(newMode)
{
    //一旦全てのコントロールを無効にする
    document.querySelector('#CropStartPosition').disabled = true;
    document.querySelector('#CropEndPosition').disabled = true;
    document.querySelector('#CropExecButton').disabled = true;

    document.querySelector('#WordSeparate1').disabled = true;
    document.querySelector('#WordSeparate2').disabled = true;
    document.querySelector('#WordSeparate3').disabled = true;
    document.querySelector('#WordSeparate4').disabled = true;
    document.querySelector('#WordSeparate5').disabled = true;
    document.querySelector('#WordSeparate6').disabled = true;
    document.querySelector('#ConvertExecButton').disabled = true;

    document.querySelector('#OutputExecButton').disabled = true;

    //WAVファイルを読み込んだ時
    if(newMode == 1)
    {
        document.querySelector('#CropStartPosition').disabled = false;
        document.querySelector('#CropEndPosition').disabled = false;
        document.querySelector('#CropExecButton').disabled = false;
    }

    //前後切り取りが完了したとき
    if(newMode == 2)
    {
        document.querySelector('#WordSeparate1').disabled = false;
        document.querySelector('#WordSeparate2').disabled = false;
        document.querySelector('#WordSeparate3').disabled = false;
        document.querySelector('#WordSeparate4').disabled = false;
        document.querySelector('#WordSeparate5').disabled = false;
        document.querySelector('#WordSeparate6').disabled = false;
        document.querySelector('#ConvertExecButton').disabled = false;
    }

    //音声変換ができたとき
    if(newMode == 3)
    {
        document.querySelector('#OutputExecButton').disabled = false;
    }

    mode = newMode;
}


//スペクトグラムを表示--------------------------------
function Disp_Spectgram()
{
    //var WavLength = 1.0 * Head_ChunkSize / Head_ByteSec; 秒数

    ctx.strokeStyle = "rgba(0,0,0,1)";
    ctx.fillText("【スペクトログラム】", 0, 158);

    let basePos = 0;
    let plotint = wave_diff.length / 1200;

    if(cropLength != -1)
    {
        basePos = cropStart;
        plotint = cropLength / 1200;
    }

    //音声データを画像に変換
        let sy = 0;

    for(let y = 0; y < 128; y++)
    {
        //描画サイズの計算
        let linelen = 3 - Math.round(3.0 * (y / 128));

        for(let a = 0; a < 1200; a++)
        {
            let p = Math.round(basePos + a * plotint);
            let tp = Math.round(y + (a * 256));

            let colB = Math.abs((Math.round(wave_diff[p] / graphZoom2)))*2;
            let colR = Math.round(dft_Re[tp] / graphZoom);
            let colG = Math.round(dft_Im[tp] / graphZoom);

            ctx.strokeStyle = "rgba(" + colR.toString() + "," + colG.toString() + "," + colB.toString() + ",1)";
            ctx.beginPath();
            ctx.moveTo(a, 160 + sy);
            ctx.lineTo(a, 160 + sy + linelen);
            ctx.stroke();
        }

        sy += linelen;    
    }
}


//微分波形を表示--------------------------------------
function Disp_DiffWave()
{
    ctx.strokeStyle = "rgba(0,0,0,1)";
    ctx.fillText("【振幅】", 0, 438);

    ctx.strokeStyle = "rgba(255,128,0,0.5)";

    let basePos = 0;
    let plotint = wave_diff.length / 1200;

    //切り取りが完了しているとき    
    if(cropLength != -1)
    {
        basePos = cropStart;
        plotint = cropLength / 1200;
    }

    let PreY = 0;

    for(let a = 0; a < 1200; a++)
    {
        let p = Math.round(basePos + a * plotint);
        let CurY = Math.round(wave_diff[p] / graphZoom2);

        ctx.beginPath();
        ctx.moveTo(a, PreY + 568);
        ctx.lineTo(a+1, CurY + 568);
        ctx.stroke();

        PreY = CurY;
    }

    //枠線を表示
    ctx.strokeStyle = "rgba(64,64,64,1)";
    ctx.beginPath();
    ctx.rect(0, 440, winWidth, 256);
    ctx.stroke();
}


//切り取りラインを表示--------------------------------
function Disp_CropLine()
{
    if(FLG_CropData == false)
    {
        ctx.strokeStyle = "rgba(255,255,0,1)";

        let p1 = Math.round(document.querySelector('#CropStartPosition').value / (src_int_Left.length / 1200));
        ctx.beginPath();
        ctx.moveTo(p1, 160);
        ctx.lineTo(p1, 696);
        ctx.stroke();

        let p2 = Math.round(document.querySelector('#CropEndPosition').value / (src_int_Left.length / 1200));
        ctx.beginPath();
        ctx.moveTo(p2, 160);
        ctx.lineTo(p2, 696);
        ctx.stroke();
    }
}


//語節切り分け位置の表示------------------------------
function Disp_SeparatePosition()
{
    ctx.strokeStyle = "rgba(255,128,255,1)";

    for(let a=0; a<6; a++)
    {
        let pt = Math.round(wordSeparate[a] / (cropLength / 1200));
        ctx.beginPath();
        ctx.moveTo(pt, 160);
        ctx.lineTo(pt, 696);
        ctx.stroke();
    }
}



//音声データ読み込み----------------------------------
function LoadSound()
{
    //フォームからファイル名の取得
    var input = document.querySelector('#SrcSoundFile').files[0];

    var reader = new FileReader();

    reader.readAsBinaryString(input);       //バイナリ配列として読み込み

    //画像オブジェクトの取得
    reader.onload = function ( event ) {

        // バイナリデータを取得
        var raw = reader.result;

        // バイナリデータを順に取得（0xffとの論理積でバイト値に変換（1））
        for (let a = 0; a < raw.length; a++){
            src_bytes[a] = raw.charCodeAt(a) & 0xff;
        }

	    //フラグ初期化
	    FLG_LoadSound = true;
        FLG_HeaderCheck = true;

        ChangeControlMode(1);

        draw();
    }

}


//ヘッダーデータチェック----------------------------------
function CheckHeaderData()
{
    //ヘッダーのチェック
    if(FLG_HeaderCheck == true)
    {
        let subBytes = [];
        var text_decoder = new TextDecoder("utf-8");
    
        //RIFF              0-3
        subBytes[0] = src_bytes[0];
        subBytes[1] = src_bytes[1];
        subBytes[2] = src_bytes[2];
        subBytes[3] = src_bytes[3];
        Head_RIFF = text_decoder.decode(Uint8Array.from(subBytes).buffer);
    
        //data              36-39
        subBytes[0] = src_bytes[36];
        subBytes[1] = src_bytes[37];
        subBytes[2] = src_bytes[38];
        subBytes[3] = src_bytes[39];
        Head_DIFdata = text_decoder.decode(Uint8Array.from(subBytes).buffer);

        //ファイルサイズ    4-7
        src_length = (src_bytes[7] << 24) + (src_bytes[6] << 16) + (src_bytes[5] << 8) + src_bytes[4];
    
        //WAVEfmt_          8-15
        subBytes[0] = src_bytes[8];
        subBytes[1] = src_bytes[9];
        subBytes[2] = src_bytes[10];
        subBytes[3] = src_bytes[11];
        subBytes[4] = src_bytes[12];
        subBytes[5] = src_bytes[13];
        subBytes[6] = src_bytes[14];
        Head_WAVEfmt = text_decoder.decode(Uint8Array.from(subBytes).buffer);
    
        //リニアPCM         16-19
        Head_LinearPCM = (src_bytes[19] << 24) + (src_bytes[18] << 16) + (src_bytes[17] << 8) + src_bytes[16];
        //フォーマット      20,21
        Head_Format =  (src_bytes[21] << 8) + src_bytes[20];
        //チャネル数        22,23
        Head_Channel =  (src_bytes[23] << 8) + src_bytes[22];
        //サンプリング      24-27
        Head_Sampling =  (src_bytes[27] << 24) + (src_bytes[26] << 16) + (src_bytes[25] << 8) + src_bytes[24];
        //秒間バイト        28-31
        Head_ByteSec =  (src_bytes[31] << 24) + (src_bytes[30] << 16) + (src_bytes[29] << 8) + src_bytes[28];
        //ブロックサイズ    32,33
        Head_BlockSize = (src_bytes[33] << 8) + src_bytes[32];
        //ビット／サンプル  34,35
        Head_BitSample = (src_bytes[35] << 8) + src_bytes[34];
        //チャンクサイズ    40-43
        Head_ChunkSize = (src_bytes[43] << 24) + (src_bytes[42] << 16) + (src_bytes[41] << 8) + src_bytes[40];      
        //以降データ        44-

        FLG_HeaderCheck = true;
    }
}


//ヘッダーデータ表示----------------------------------
function Disp_Header()
{
    //ヘッダーの表示
    ctx.fillText("【WAVファイルヘッダー】", 0, 60);
    ctx.fillText("[" + Head_RIFF + "] [FileSize " + src_length.toString() + "] [" + Head_WAVEfmt + "] [" + Head_LinearPCM.toString() + "] [" +
                      Head_Format.toString() + "] [Channels " + Head_Channel.toString() + "] [" + Head_Sampling.toString() + " Hz] [" + 
                      Head_ByteSec.toString() + " bps] [BlockSize " + Head_BlockSize.toString() + "] [BitPerSamp " +
                      Head_BitSample.toString() + "] [" + Head_DIFdata + " " + Head_ChunkSize.toString() + "]", 20, 120);
}

//開始～終了位置を変更する----------------------------------
function CropPosition(num)
{
    if(document.querySelector('#CropStartPosition').value > document.querySelector('#CropEndPosition').value)
    {
        if(num == 0)
        {
            //開始側を変更したとき
            document.querySelector('#CropEndPosition').value = document.querySelector('#CropStartPosition').value
        }

        if(num == 1)
        {
            //終了側を変更したとき
            document.querySelector('#CropStartPosition').value = document.querySelector('#CropEndPosition').value
        }
    }

    document.querySelector('#output1').value = document.querySelector('#CropStartPosition').value;
    document.querySelector('#output2').value = document.querySelector('#CropEndPosition').value;

    draw();
}


//語節切り分け位置を変更する--------------------------
function SeparatePosition()
{
    wordSeparate[0] = Number(document.querySelector('#WordSeparate1').value);
    wordSeparate[1] = Number(document.querySelector('#WordSeparate2').value);
    wordSeparate[2] = Number(document.querySelector('#WordSeparate3').value);
    wordSeparate[3] = Number(document.querySelector('#WordSeparate4').value);
    wordSeparate[4] = Number(document.querySelector('#WordSeparate5').value);
    wordSeparate[5] = Number(document.querySelector('#WordSeparate6').value);

    document.querySelector('#outputS1').value = document.querySelector('#WordSeparate1').value;
    document.querySelector('#outputS2').value = document.querySelector('#WordSeparate2').value;
    document.querySelector('#outputS3').value = document.querySelector('#WordSeparate3').value;
    document.querySelector('#outputS4').value = document.querySelector('#WordSeparate4').value;
    document.querySelector('#outputS5').value = document.querySelector('#WordSeparate5').value;
    document.querySelector('#outputS6').value = document.querySelector('#WordSeparate6').value;

    draw();
}



//データをbyte⇒int変換する---------------------------
function ConvertDataByteToInt()
{
    let Interval = Head_Channel * 4;
    let SampNum = Head_ChunkSize / Interval;        //サンプリングデータの数

    for(let a=0; a<SampNum; a++)
    {
        let Pos = (a * Interval) + 44;
        let Val_L = (((src_bytes[Pos+3] << 24) + (src_bytes[Pos+2] << 16) + (src_bytes[Pos+1] << 8) + src_bytes[Pos])) / 2;
        let Val_R = (((src_bytes[Pos+7] << 24) + (src_bytes[Pos+6] << 16) + (src_bytes[Pos+5] << 8) + src_bytes[Pos+4]) / 2);
        src_int_Left[a] = Val_L;
        src_int_Right[a] = Val_R;
    }   
}


//平均値を求める----------------------------------------
function CulcMean()
{
    let meanLeft = new Array(src_int_Left.length);
    let meanRight = new Array(src_int_Right.length);

    for(let a=0; a<src_int_Left.length - 2; a++)
    {
        meanLeft[a] = (src_int_Left[a] + src_int_Left[a+1]+ src_int_Left[a+2]) / 3;
        meanRight[a] = (src_int_Right[a] + src_int_Right[a+1] + src_int_Right[a+2]) / 3;
    }

    src_int_Left = meanLeft.slice();
    src_int_Right = meanRight.slice();
}


//生データの微分を行う---------------------------------
function CalcDiff()
{
    let tmpMax = 0;

    for(let a=0; a<src_int_Left.length - 1; a++)
    {
        let val1 = (src_int_Left[a] + src_int_Right[a]) / 2;
        let val2 = (src_int_Left[a+1] + src_int_Right[a+1]) / 2;
        let val3 = val2 - val1;

        if(tmpMax < Math.abs(val3))
        {
            tmpMax = Math.abs(val3);
        }

        wave_diff.push(val3);
    }

    graphZoom2 = tmpMax / 128;
}


//短時間フーリエ変換を実行する------------------------
function Calc_SDFT()
{
    graphZoom = 1;

    //DFTで計算
    //for(let a=0; a<1200; a++)
    //{
        //short_dft(a, 1, 256);
    //}

    //FFTで計算
    for(let a=0; a<1200; a++)
    {
        fftmain(a, 256);
    }


    FLG_CalcDFT = true;

    if(FLG_CropData == false)
    {
        cropStart = 0;
        cropEnd = src_int_Left.length;
        cropLength = src_int_Left.length;

        //データ切り取り未
        document.querySelector('#CropStartPosition').max = src_int_Left.length;
        document.querySelector('#CropStartPosition').value = 0;
        document.querySelector('#CropEndPosition').max = src_int_Left.length;
        document.querySelector('#CropEndPosition').value = src_int_Left.length;

        //\\\\\debug
        document.querySelector('#CropStartPosition').value = 19800;
        document.querySelector('#CropEndPosition').value = 42200;

        document.querySelector('#output1').value = document.querySelector('#CropStartPosition').value;
        document.querySelector('#output2').value = document.querySelector('#CropEndPosition').value;
    }
}


//フーリエ変換 ---------------------------------------
/*
function dft()
{
    let stepNum = Math.round(src_int_Left.length / 1200);

    let N = 1200;

    for(let a=0; a<N; ++a)
    {
        let Re_sum = 0.0;
        let Im_sum = 0.0;

        for(let b=0; b<N; ++b )
        {
            var tht = 2 * Math.PI/N * a * b;

            Re_sum += (src_int_Left[(b * stepNum)] * Math.cos( tht ) + src_int_Right[(b * stepNum)] * Math.cos( tht )) /2;
            Im_sum += (src_int_Left[(b * stepNum)] * Math.sin( tht ) + src_int_Right[(b * stepNum)] * Math.sin( tht )) /2;
        }

        dft_Re_Wave.push( Re_sum );
        dft_Im_Wave.push( Im_sum );
    }
}
*/


//短時間フーリエ変換 ---------------------------------------
function short_dft(frameNum, stepNum, sampNum)
{
    let tmpMax = 0;

    let basePos = 0;
    let startPos = Math.round(src_int_Left.length / 1200) * frameNum;

    //切り取りが完了しているとき
    if(cropLength != -1)
    {
        basePos = cropStart;
        startPos = Math.round(cropLength / 1200) * frameNum;
    }

    //let adjust = startPos % 4;
    //startPos -= adjust;

    for(let a=0; a<sampNum; ++a)
    {
        let Re_sum = 0.0;
        let Im_sum = 0.0;

        for(let b=0; b<sampNum; ++b )
        {
            var tht = 2.0 * Math.PI/sampNum * a * b;

            Re_sum += (src_int_Left[basePos + startPos + (b * stepNum)] * Math.cos( tht ) +
                       src_int_Right[basePos + startPos + (b * stepNum)] * Math.cos( tht )) /2;
            Im_sum += (src_int_Left[basePos + startPos + (b * stepNum)] * Math.sin( tht ) +
                       src_int_Right[basePos + startPos + (b * stepNum)] * Math.sin( tht )) /2;
        }

        if(tmpMax < Math.abs(Re_sum))
        {
            tmpMax = Math.abs(Re_sum);
        }

        dft_Re.push(Re_sum);
        dft_Im.push(Im_sum);
    }

    if(graphZoom < tmpMax / 256 /16)
    {
        graphZoom = tmpMax / 256 / 16;
    }
}



//高速フーリエ変換(再帰部) -----------------------------------
function fftrec(dataArray, T, N, s = 0, w = 1)
{
    //dataArray データの並び(メイン配列から必要な要素数をコピーしたもの)

    if(N === 1) return [dataArray[s]];

    const Nh = N / 2;
    const Td = T * 2;
    const wd = w * 2;

    const rec = fftrec(dataArray, Td, Nh, s, wd).concat(fftrec(dataArray, Td, Nh, s + w, wd));

    for(let i=0; i < Nh; i++)
    {
        const l = rec[i];
        const re = imul(rec[i + Nh], expi(T * i));

        [rec[i], rec[i + Nh]] = [iadd(l, re), isub(l, re)];
    }

    return rec;
}

function expi(theta) {return [Math.cos(theta), Math.sin(theta)];}
function iadd([ax, ay], [bx, by]) {return [ax + bx, ay + by];}
function isub([ax, ay], [bx, by]) {return [ax - bx, ay - by];}
function imul([ax, ay], [bx, by]) {return [(ax * bx) - (ay * by), (ax * by) + (ay * bx)];}
function isum(cs) {return cs.reduce((s, c) => iadd(s, c), [0, 0]);}

//高速フーリエ変換(実行部) -----------------------------------
function fftmain(frameNum, Length)
{
    let tmpMax = 0;

    let basePos = 0;
    let startPos = Math.round(src_int_Left.length / 1200) * frameNum;

    //切り取りが完了しているとき
    if(cropLength != -1)
    {
        basePos = cropStart;
        startPos = Math.round(cropLength / 1200) * frameNum;
    }

    //let adjust = startPos % 4;
    //startPos -= adjust;

    const T = -2 * Math.PI / Length;

    const dataPartLeft = src_int_Left.slice(basePos + startPos, basePos + startPos + Length);
    const dataPartRight = src_int_Right.slice(basePos + startPos, basePos + startPos + Length);

    if(dataPartLeft.length < 256) 
    {
        return;
    }

    //窓関数適用（ハン窓）
    const dataPartLeft2 = hann_window(dataPartLeft, 256);
    const dataPartRight2 = hann_window(dataPartRight, 256);  

    const f0Left = dataPartLeft2.map(r => [r, 0]);
    const f0Right = dataPartRight2.map(r => [r, 0]);

    const resultLeft = fftrec(f0Left, T, f0Left.length);
    const resultRight = fftrec(f0Right, T, f0Right.length);

    const Re_Left = resultLeft.map(([r, i]) => r);
    const Im_Left = resultLeft.map(([r, i]) => i);
    const Re_Right = resultRight.map(([r, i]) => r);
    const Im_Right = resultRight.map(([r, i]) => i);

    for(let a=0; a<Length; a++)
    {
        let Re_sum = (Re_Left[a] + Re_Right[a] ) / 2;
        let Im_sum = (Im_Left[a] + Im_Right[a] ) / 2;

        if(tmpMax < Math.abs(Re_sum))
        {
            tmpMax = Math.abs(Re_sum);
        }

        dft_Re.push(Re_sum);
        dft_Im.push(Im_sum);
    }

    if(graphZoom < tmpMax / 256 /16)
    {
        graphZoom = tmpMax / 256 / 16;
    }
}


//窓関数（ハン窓）------------------------------------
function hann_window(dataArray, Length)
{
    const ret = new Array(Length);

    for(let a=0; a<Length; a++)
    {
        let tht = 2.0 * Math.PI * (1.0 * a / Length);
        ret[a] = dataArray[a] * (0.5 - 0.5 * Math.cos(tht));
    }

    return ret;
}


//切り取り実行----------------------------------------
function CropDataExec()
{
    cropStart = Number(document.querySelector('#CropStartPosition').value);
    cropEnd = Number(document.querySelector('#CropEndPosition').value);
    cropLength = cropEnd - cropStart;
    backLength = src_int_Left.length - cropEnd;

    //語節切り分けスライダのセット
    document.querySelector('#WordSeparate1').max = cropLength;
    document.querySelector('#WordSeparate2').max = cropLength;
    document.querySelector('#WordSeparate3').max = cropLength;
    document.querySelector('#WordSeparate4').max = cropLength;
    document.querySelector('#WordSeparate5').max = cropLength;
    document.querySelector('#WordSeparate6').max = cropLength;

    let sepPos = Math.round(cropLength / 7);
    document.querySelector('#WordSeparate1').value = sepPos;
    document.querySelector('#WordSeparate2').value = sepPos * 2;
    document.querySelector('#WordSeparate3').value = sepPos * 3;
    document.querySelector('#WordSeparate4').value = sepPos * 4;
    document.querySelector('#WordSeparate5').value = sepPos * 5;
    document.querySelector('#WordSeparate6').value = sepPos * 6;
    
    //\\\\\debug
    document.querySelector('#WordSeparate1').value = 1680;
    document.querySelector('#WordSeparate2').value = 6440;
    document.querySelector('#WordSeparate3').value = 9820;
    document.querySelector('#WordSeparate4').value = 13200;
    document.querySelector('#WordSeparate5').value = 16780;
    document.querySelector('#WordSeparate6').value = 20320;

    FLG_CropData = true;

    //SDFTをやり直し
    dft_Re.splice(0);
    dft_Im.splice(0);
    FLG_CalcDFT = false;

    ChangeControlMode(2);

    SeparatePosition();
}


//音声合成実行----------------------------------------
function ConvertDataExec()
{
    //ヘッダー部のコピー
    for(let a=0; a<44; a++)
    {
        dst_bytes[a] = src_bytes[a];
    }

    //現状の書き込み位置
    let DataStep = 8;                   //１データ当たりのバイト数    

    //使用する長さ情報の計算
    let SL1 = wordSeparate[1] - wordSeparate[0];    //し
    let SL2 = wordSeparate[2] - wordSeparate[1];    //ま
    let SL3 = wordSeparate[3] - wordSeparate[2];    //え
    let SL4 = wordSeparate[4] - wordSeparate[3];    //な
    let SL5 = wordSeparate[5] - wordSeparate[4];    //が
    let SA = Math.round(SL4 / 2);                   //なの半分の長さ


    //--------------------------------------------------
    //データチャンク[1]  開始位置～最初の文字の手前
    let SPS = 0;
    let SPD = 0;
    let LEN = cropStart;
    WAVSet_Part(SPS * DataStep, SPD * DataStep, LEN * DataStep);

    //--------------------------------------------------
    //データチャンク[2]  切り取り先頭～しの前まで     ...元データをそのままコピー
    SPS = cropStart;
    SPD = cropStart;
    LEN = wordSeparate[0];
    WAVSet_Part(SPS * DataStep, SPD * DataStep, LEN * DataStep);

    //--------------------------------------------------
    //データチャンク[3]  し     ...元データをそのままコピー
    SPS = cropStart + wordSeparate[0];
    SPD = cropStart + wordSeparate[0];
    LEN = SL1;
    WAVSet_Part(SPS * DataStep, SPD * DataStep, LEN * DataStep);

    //--------------------------------------------------
    //データチャンク[4]  ら     ...合成して作成
    SPS = cropStart + wordSeparate[1];
    SPD = cropStart + wordSeparate[1];
    LEN = SL2;
    WAVSet_Part(SPS * DataStep, SPD * DataStep, LEN * DataStep);

    //--------------------------------------------------
    //データチャンク[5]  ん     ...元データの「な」の前半部分をコピー
    SPS = cropStart + wordSeparate[3];
    SPD = cropStart + wordSeparate[2];
    LEN = SA;
    WAVSet_Part(SPS * DataStep, SPD * DataStep, LEN * DataStep);

    //--------------------------------------------------
    //データチャンク[6]  が     ...元データの５文字目「が」の部分をコピー
    SPS = cropStart + wordSeparate[4];
    SPD = cropStart + wordSeparate[2] + SA;
    LEN = SL5;
    WAVSet_Part(SPS * DataStep, SPD * DataStep, LEN * DataStep);

    //--------------------------------------------------
    //データチャンク[7]  な     ...元データの４文字目「な」の部分をコピー
    SPS = cropStart + wordSeparate[3];
    SPD = cropStart + wordSeparate[2] + SA + SL5;
    LEN = SL4;
    WAVSet_Part(SPS * DataStep, SPD * DataStep, LEN * DataStep);

    //--------------------------------------------------
    //データチャンク[8]  切り取り末尾まで     ...元データの５文字目から切り取り末尾まで
    SPS = cropStart + wordSeparate[5];
    SPD = cropStart + wordSeparate[2] + SA + SL5 + SL4;
    LEN = cropLength - wordSeparate[5];
    WAVSet_Part(SPS * DataStep, SPD * DataStep, LEN * DataStep);

    //--------------------------------------------------
    //データチャンク[9]  最後の部分まで
    SPS = cropStart + cropLength;
    SPD = cropStart + wordSeparate[2] + SA + SL5 + SL4 + (cropLength - wordSeparate[5]);
    LEN = backLength;
    WAVSet_Part(SPS * DataStep, SPD * DataStep, LEN * DataStep);

    //--------------------------------------------------
    //データチャンク長さを書き換える
    Set_ChunkLength(Head_ChunkSize - SA * DataStep);

    //処理完了
    FLG_ConvExec = false;
    ChangeControlMode(3);
    draw();
}


//WAVデータのセット(そのままコピー) --------------------------------------
function WAVSet_Part(SrcPos, DstPos, CpLen)
{
    let ChunkStart = 44;

    for(let a=0; a<CpLen; a++)
    {
        let posS = ChunkStart + SrcPos + a;
        let posD = ChunkStart + DstPos + a;
        dst_bytes[posD] = src_bytes[posS];
    }
}


//チャンクサイズを更新する
function Set_ChunkLength(NewLen)
{
    let TmpArray = new Uint8Array([
        (NewLen >> 24) & 255,
        (NewLen >> 16) & 255,
        (NewLen >> 8) & 255,
        NewLen & 255,
      ]);

    //チャンクサイズ    40-43
    dst_bytes[43] = TmpArray[0];
    dst_bytes[42] = TmpArray[1];
    dst_bytes[41] = TmpArray[2];
    dst_bytes[40] = TmpArray[3];
}


//音声出力----------------------------------------
function OutputWAVData()
{
    var BinaryArray = new Uint8Array(dst_bytes);
    var audioBlob = new Blob([BinaryArray], { type: 'audio/wav' });
    //var myURL = window.URL || window.webkitURL;
    //var url = myURL.createObjectURL(audioBlob);

    const downloadLink = document.getElementById('DstDownload');
    downloadLink.href = window.URL.createObjectURL(audioBlob);
    downloadLink.download = 'Shirangana.wav';
    downloadLink.click();

    //return url;
}


//最初に実行------------------------------------------
function main(){
    init();
    //draw();
}

function MM_closeFloatWindow(theURL) { //v2.0
    window.close(this);
}

